# -*- coding: utf-8 -*-

import base64
from datetime import date
from io import BytesIO as StringIO

import xlsxwriter

from odoo import models, fields


class AccountAssetAccountingReport(models.TransientModel):
    _name = 'account.asset.accounting.report'

    report_name = fields.Char(string='Report Name', default="Asset Accounting Report.xlsx")
    as_at_date = fields.Date(string='As At Date', required=True)
    data = fields.Binary(string='Download File', readonly=True)
    lines = fields.One2many(comodel_name='account.asset.accounting.report.line',
                            inverse_name='wizard_id', string="Lines")

    def calculate_year_start_date(self):
        self.ensure_one()
        year_end_month = int(self.env.company.fiscalyear_last_month)
        if year_end_month == 12:
            month = 1
        else:
            month = year_end_month + 1

        if self.as_at_date.month > year_end_month:
            year = self.as_at_date.year
        else:
            year = self.as_at_date.year - 1

        return date(year, month, 1)

    def build_data(self):
        self.ensure_one()

        # Clear out existing.
        self.lines.unlink()

        wizard = self
        asset_ids = self.env['account.asset'].search([('company_id', '=', self.env.company.id), ('state', 'not in', ['draft', 'model'])])
        for asset in asset_ids:
            addition = opening_balance = disposal = depn_opening = depreciation = withdrawn = sale_value = gain_loss_on_disposal = 0

            sold = False
            year_start_date = self.calculate_year_start_date()

            if asset.acquisition_date > wizard.as_at_date:
                continue
            if asset.disposal_date and asset.disposal_date < year_start_date:
                continue
            if asset.acquisition_date < year_start_date:
                opening_balance = asset.original_value
            else:
                addition = asset.original_value
            if asset.disposal_date and year_start_date <= asset.disposal_date <= wizard.as_at_date:
                disposal = asset.original_value
                sold = True
                sale_value = asset.sale_amount

            moves = asset.depreciation_move_ids.filtered(lambda x: x.state == 'posted' and x.date <= self.as_at_date)
            # TODO needs improving
            if moves:
                last_move_date = moves[0].date
                last_move = moves[0]
                for move in moves:
                    if move.date >= last_move_date and move.id > last_move.id:
                        last_move_date = move.date
                        last_move = move

                for move in moves:
                    depn_exp_lines = move.line_ids.filtered(lambda x: not x.account_id.user_type_id.include_initial_balance)
                    if sold and move.id == last_move.id:
                        # want to split any gain/loss on sale into separate column so find the last move and use depn_exp_lines from that
                        # as the gain or loss entry
                        gain_loss_on_disposal = sum([x.debit - x.credit for x in depn_exp_lines])
                    elif move.date < year_start_date:
                        depn_opening = sum([x.debit - x.credit for x in depn_exp_lines])
                    else:
                        depreciation = sum([x.debit - x.credit for x in depn_exp_lines])

            if sold:
                withdrawn = depn_opening + depreciation

            self.env['account.asset.accounting.report.line'].create({
                'wizard_id': self.id,
                'asset_id': asset.id,
                'asset_category_id': asset.account_asset_id.id,
                'cost_price_opening': opening_balance,
                'addition': addition,
                'disposal': disposal,
                'depn_opening': depn_opening,
                'depreciation': depreciation,
                'withdrawn': withdrawn,
                'gain_loss_on_disposal': gain_loss_on_disposal,
                'sale_value': sale_value

            })

    def format_date(self, as_at_date):
        day = as_at_date.day
        if day < 10:
            day = '0' + str(as_at_date.day)
        else:
            day = str(as_at_date.day)

        month = as_at_date.month
        if month < 10:
            month = '0' + str(as_at_date.month)
        else:
            month = str(as_at_date.month)

        formatted_date = day + '/' + month + '/' + str(as_at_date.year)
        return formatted_date

    def button_process(self):
        """ Create the report. """
        self.ensure_one()
        wizard = self.browse(self.ids[0])
        data = StringIO()

        # build the data table
        self.build_data()

        workbook = xlsxwriter.Workbook(data)
        worksheet = workbook.add_worksheet('Data')
        format_number = workbook.add_format({'num_format': '#,##0.00', 'align': 'right'})
        format_row = workbook.add_format({'text_wrap': True, 'bold': True, 'size': 12})
        format_row_right = workbook.add_format({'text_wrap': True, 'bold': True, 'size': 12, 'align': 'right'})
        format_row_no_wrap = workbook.add_format({'bold': True})
        format_row_centre = workbook.add_format({'bold': True, 'align': 'center'})
        format_total_cell = workbook.add_format({'bold': True, 'num_format': '#,##0.00', 'align': 'right'})

        date_for_report = self.format_date(wizard.as_at_date)

        row = 0

        # write report heading
        worksheet.set_column('J:J', 3)
        worksheet.set_column('O:O', 3)
        worksheet.set_column('Q:Q', 3)
        worksheet.write(0, 0, 'Asset Report - Accounting - As At ' + date_for_report, format_row_no_wrap)
        row += 1

        worksheet.merge_range(row, 5, row, 8, '<------------------ Cost Price ----------------->', format_row_centre)
        worksheet.merge_range(row, 10, row, 14, '<----------------------- Depreciation Provision---------------------->', format_row_centre)
        row += 1

        worksheet.write(row, 0, 'Code', format_row)
        worksheet.write(row, 1, 'Description', format_row)
        worksheet.write(row, 2, 'Serial #', format_row)
        worksheet.write(row, 3, 'Purchase Date', format_row)
        worksheet.write(row, 4, 'Disposal Date', format_row)
        worksheet.write(row, 5, 'Opening Balance', format_row_right)
        worksheet.write(row, 6, 'Additions', format_row_right)
        worksheet.write(row, 7, 'Disposals', format_row_right)
        worksheet.write(row, 8, 'Closing Balance', format_row_right)
        worksheet.write(row, 10, 'Opening Balance', format_row_right)
        worksheet.write(row, 11, 'Depreciation', format_row_right)
        worksheet.write(row, 12, 'Withdrawn', format_row_right)
        worksheet.write(row, 13, 'Closing Balance', format_row_right)
        worksheet.write(row, 15, 'Book Value', format_row_right)
        worksheet.write(row, 17, 'Sale Value', format_row_right)
        worksheet.write(row, 18, '(Gain)/Loss', format_row_right)
        row += 1

        asset_opening_total = asset_addition_total = asset_disposal_total \
            = depn_opening_total = depn_depreciation_total = depn_withdrawn_total = gain_loss_total = asset_sale_value_total = 0

        categories = list(set([x.asset_category_id for x in self.lines]))
        for category in categories:
            row += 2
            worksheet.write(row, 0, category.name, format_row_no_wrap)
            row += 1
            asset_opening_cat_total = asset_addition_cat_total = asset_disposal_cat_total \
                = depn_opening_cat_total = depn_depreciation_cat_total = depn_withdrawn_cat_total \
                = depn_gain_loss_cat_total = asset_sale_value_cat_total = 0
            assets = [x for x in self.lines if x.asset_category_id == category]
            for asset in assets:
                if asset.asset_id.disposal_date:
                    disposal_date = self.format_date(asset.asset_id.disposal_date)
                else:
                    disposal_date = ''

                worksheet.write(row, 0, asset.asset_id.code)
                worksheet.write(row, 1, asset.asset_id.name)
                worksheet.write(row, 2, asset.asset_id.serial_number or '')
                worksheet.write(row, 3, self.format_date(asset.asset_id.acquisition_date))
                worksheet.write(row, 4, disposal_date)
                worksheet.write(row, 5, asset.cost_price_opening, format_number)
                worksheet.write(row, 6, asset.addition, format_number)
                worksheet.write(row, 7, asset.disposal, format_number)
                worksheet.write(row, 8, asset.cost_price_opening + asset.addition - asset.disposal, format_number)
                worksheet.write(row, 10, asset.depn_opening, format_number)
                worksheet.write(row, 11, asset.depreciation, format_number)
                worksheet.write(row, 12, asset.withdrawn, format_number)
                worksheet.write(row, 13, asset.depn_opening + asset.depreciation - asset.withdrawn, format_number)
                worksheet.write(row, 15, (asset.cost_price_opening + asset.addition - asset.disposal) -
                                (asset.depn_opening + asset.depreciation - asset.withdrawn), format_number)
                worksheet.write(row, 17, asset.sale_value, format_number)
                worksheet.write(row, 18, asset.gain_loss_on_disposal, format_number)
                row += 1

                asset_opening_cat_total += asset.cost_price_opening
                asset_addition_cat_total += asset.addition
                asset_disposal_cat_total += asset.disposal
                depn_opening_cat_total += asset.depn_opening
                depn_depreciation_cat_total += asset.depreciation
                depn_withdrawn_cat_total += asset.withdrawn
                depn_gain_loss_cat_total += asset.gain_loss_on_disposal
                asset_sale_value_cat_total += asset.sale_value

                asset_opening_total += asset.cost_price_opening
                asset_addition_total += asset.addition
                asset_disposal_total += asset.disposal
                depn_opening_total += asset.depn_opening
                depn_depreciation_total += asset.depreciation
                depn_withdrawn_total += asset.withdrawn
                gain_loss_total += asset.gain_loss_on_disposal
                asset_sale_value_total += asset.sale_value

            worksheet.write(row, 5, asset_opening_cat_total, format_total_cell)
            worksheet.write(row, 6, asset_addition_cat_total, format_total_cell)
            worksheet.write(row, 7, asset_disposal_cat_total, format_total_cell)
            worksheet.write(row, 8, asset_opening_cat_total + asset_addition_cat_total - asset_disposal_cat_total,
                            format_total_cell)
            worksheet.write(row, 10, depn_opening_cat_total, format_total_cell)
            worksheet.write(row, 11, depn_depreciation_cat_total, format_total_cell)
            worksheet.write(row, 12, depn_withdrawn_cat_total, format_total_cell)
            worksheet.write(row, 13, depn_opening_cat_total + depn_depreciation_cat_total - depn_withdrawn_cat_total,
                            format_total_cell)
            worksheet.write(row, 15, asset_opening_cat_total + asset_addition_cat_total - asset_disposal_cat_total -
                            (depn_opening_cat_total + depn_depreciation_cat_total - depn_withdrawn_cat_total),
                            format_total_cell)
            worksheet.write(row, 17, asset_sale_value_total, format_total_cell)
            worksheet.write(row, 18, depn_gain_loss_cat_total, format_total_cell)

            row += 2

        worksheet.write(row, 1, 'Grand Total', format_row_no_wrap)
        worksheet.write(row, 5, asset_opening_total, format_total_cell)
        worksheet.write(row, 6, asset_addition_total, format_total_cell)
        worksheet.write(row, 7, asset_disposal_total, format_total_cell)
        worksheet.write(row, 8, asset_opening_total + asset_addition_total - asset_disposal_total, format_total_cell)
        worksheet.write(row, 10, depn_opening_total, format_total_cell)
        worksheet.write(row, 11, depn_depreciation_total, format_total_cell)
        worksheet.write(row, 12, depn_withdrawn_total, format_total_cell)
        worksheet.write(row, 13, depn_opening_total + depn_depreciation_total - depn_withdrawn_total, format_total_cell)
        worksheet.write(row, 15, asset_opening_total + asset_addition_total - asset_disposal_total -
                        (depn_opening_total + depn_depreciation_total - depn_withdrawn_total), format_total_cell)
        worksheet.write(row, 17, asset_sale_value_total, format_total_cell)
        worksheet.write(row, 18, gain_loss_total, format_total_cell)

        worksheet.set_column('A:A', 8)
        worksheet.set_column('B:B', 30)
        worksheet.set_column('C:C', 8)
        worksheet.set_column('D:D', 16)
        worksheet.set_column('E:I', 14)
        worksheet.set_column('K:N', 14)
        worksheet.set_column('P:P', 14)
        worksheet.set_column('R:S', 14)

        workbook.close()
        data.seek(0)
        output = base64.encodestring(data.read())
        self.data = output
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'account.asset.accounting.report',
            'view_mode': 'form',
            'view_type': 'form',
            'res_id': wizard.id,
            'target': 'new', }


class account_asset_accounting_report_line(models.TransientModel):
    _name = 'account.asset.accounting.report.line'

    wizard_id = fields.Many2one(string='Wizard', comodel_name='account.asset.accounting.report')
    asset_id = fields.Many2one(string='Asset', comodel_name='account.asset')
    asset_category_id = fields.Many2one(string="Category", comodel_name='account.account')
    cost_price_opening = fields.Float(string='Cost Price Opening')
    addition = fields.Float(string='Addition')
    disposal = fields.Float(string='Disposal')
    depn_opening = fields.Float(string='Opening Depreciation')
    depreciation = fields.Float(string='Depreciation')
    withdrawn = fields.Float('Withdrawn')
    sale_value = fields.Float('Sale Value')
    gain_loss_on_disposal = fields.Float('Gain Loss on Disposal')
